
#include <nxp/iolpc1766.h>
#include "arm_comm.h"
#include <math.h>

void InitClock(void);

#define WAVE_SAMPLE_NUM       64
#define PI                    (float)(3.141592654)

static inline void __WFI()    { asm ("wfi"); }

extern signed long sinusoide[WAVE_SAMPLE_NUM];

extern struct {
  Int32U  source;     // start of source area
  Int32U  destination;// start of destination area
  Int32U  next;       // address of next strLLI in chain
  Int32U  control;    // DMACCxControl register
} LLI0;


/* Main */
int main (void)
{
    int i;

    InitClock(); 
    
    for(i=0; i < WAVE_SAMPLE_NUM; i++)
        sinusoide[i] = 512*sin(2*PI*i/WAVE_SAMPLE_NUM);

    for(i=0; i < WAVE_SAMPLE_NUM; i++)
        sinusoide[i] = ((sinusoide[i] + 512) << 6)// DACR bit 6-15, VALUE
		             | 1 << 16;// DACR bit 16, BIAS = 1

    /* Enable AOUT (P0.26) */
    PINSEL1 |= (2<<20);
    
    /* Configure output LEDs */
    FIO1DIR |=  (1<<28) | (1<<29);
    FIO1CLR  =   (1<<28);
    FIO1SET  =   (1<<29);

    /* Configure LLI structure */
    LLI0.source      = (Int32U) &sinusoide[0];
    LLI0.destination = (Int32U) &(DACR);
    LLI0.next        = (Int32U) &LLI0;
    LLI0.control     = 1<<26 | 2<<21 | 2<<18 | WAVE_SAMPLE_NUM;

    /* Power up the GPDMA */
    PCONP |= (1<<29);
    
    /* Enable GPDMA  and sync logic */
    DMACCONFIGURATION = 1;
    DMACSYNC          = (1<<6); 
    
    /* Load DMA Channel0 */
    DMACC0SRCADDR   = (Int32U) &sinusoide[0];
    DMACC0DESTADDR  = (Int32U) &(DACR);
    DMACC0LLI       = (Int32U) &LLI0;
    
    DMACC0CONTROL = WAVE_SAMPLE_NUM   // transfer size (0 - 11) = 64
                  | (0 << 12)         // source burst size (12 - 14) = 1
                  | (0 << 15)         // destination burst size (15 - 17) = 1
                  | (2 << 18)         // source width (18 - 20) = 32 bit
                  | (2 << 21)         // destination width (21 - 23) = 32 bit
                  | (0 << 24)         // source AHB select (24) = AHB 0
                  | (0 << 25)         // destination AHB select (25) = AHB 0
                  | (1 << 26)         // source increment (26) = increment
                  | (0 << 27)         // destination increment (27) = no increment
                  | (0 << 28)         // mode select (28) = access in user mode
                  | (0 << 29)         // (29) = access not bufferable
                  | (0 << 30)         // (30) = access not cacheable
                  | (0 << 31);        // terminal count interrupt disabled

    DMACC0CONFIGURATION = 1
                  | (0 << 1) 	      // source peripheral (1 - 5) = none
                  | (7 << 6) 	      // destination peripheral (6 - 10) = DAC
                  | (1 << 11)	      // flow control (11 - 13) = mem to per
                  | (0 << 14)	      // (14) = mask out error interrupt
                  | (0 << 15)	      // (15) = mask out terminal count interrupt
                  | (0 << 16)	      // (16) = no locked transfers
                  | (0 << 18);        // (27) = no HALT


    /* DACclk = 25 MHz, so 10 usec interval	*/
    DACCNTVAL = 250;
    /* DMA, timer running, dbuff */
    DACCTRL = 1<<3 | 1<<2 | 1<<1; 


    i = 0;
    while(1)    // Loop forever
    {
	/* Toggle LED to indicate that CPU is running */
	if(i == 0xF0000){
          FIO1PIN ^= (1<<28) | (1<<29);
	  i = 0;
          /* If button INT0 is pressed, turn off LEDs and go to sleep */
      	  if(!(FIO2PIN & (1<<10))){
            FIO1CLR = (1<<28) | (1<<29);
            __WFI();
	  }
	}else
	  i++;
					
    }
}




/*************************************************************************
 * Function Name: InitClock
 * Parameters: void
 * Return: void
 *
 * Description: Initialize PLL and clocks' dividers. Hclk - 300MHz,
 *              Fcckl = 100MHz
 *
 *************************************************************************/
void InitClock(void)
{
  // 1. Init OSC
  SCS_bit.OSCRANGE = 0;
  SCS_bit.OSCEN = 1;
  // 2.  Wait for OSC ready
  while(!SCS_bit.OSCSTAT);
  // 3. Disconnect PLL
  PLL0CON_bit.PLLC = 0;
  PLL0FEED = 0xAA;
  PLL0FEED = 0x55;
  // 4. Disable PLL
  PLL0CON_bit.PLLE = 0;
  PLL0FEED = 0xAA;
  PLL0FEED = 0x55;
  // 5. Select source clock for PLL
  CLKSRCSEL_bit.CLKSRC = 1; // Selects the main oscillator as a PLL clock source.
  // 6. Set PLL settings 300 MHz
  PLL0CFG_bit.MSEL = 25-1;
  PLL0CFG_bit.NSEL = 2-1;
  PLL0FEED = 0xAA;
  PLL0FEED = 0x55;
  // 7. Enable PLL
  PLL0CON_bit.PLLE = 1;
  PLL0FEED = 0xAA;
  PLL0FEED = 0x55;
  // 8. Wait for the PLL to achieve lock
  while(!PLL0STAT_bit.PLOCK);
  // 9. Set clk divider settings
  CCLKCFG   = 3-1;            // 1/3 Fpll
  PCLKSEL0 = PCLKSEL1 = 0;    // other peripherals 100/4 = 25MHz
  // 10. Connect the PLL
  PLL0CON_bit.PLLC = 1;
  PLL0FEED = 0xAA;
  PLL0FEED = 0x55;
}